<?php
/* --------------------------------------------------------------
   CreateParcelServicesAction.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\ParcelService\Actions;

use Gambio\Admin\ParcelService\Interfaces\ParcelServiceId;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceService;
use Gambio\Admin\ParcelService\Model\ParcelService;
use Gambio\Admin\ParcelService\Model\ParcelServices;
use Gambio\Api\Application\Responses\CreateApiMetaDataTrait;
use Gambio\Api\ParcelService\ParcelServiceApiRequestParser;
use Gambio\Api\ParcelService\ParcelServiceApiRequestValidator;
use InvalidArgumentException;
use Slim\Http\Response;
use Slim\Http\ServerRequest as Request;

/**
 * Class CreateParcelServicesAction
 *
 * @package Gambio\Api\ParcelService\Actions
 */
class CreateParcelServicesAction
{
    use CreateApiMetaDataTrait;
    
    /**
     * @var ParcelServiceApiRequestParser
     */
    private $requestParser;
    
    /**
     * @var ParcelServiceApiRequestValidator
     */
    private $requestValidator;
    
    /**
     * @var ParcelServiceService
     */
    private $service;
    
    
    /**
     * CreateParcelServicesAction constructor.
     *
     * @param ParcelServiceApiRequestParser    $requestParser
     * @param ParcelServiceApiRequestValidator $requestValidator
     * @param ParcelServiceService             $service
     */
    public function __construct(
        ParcelServiceApiRequestParser $requestParser,
        ParcelServiceApiRequestValidator $requestValidator,
        ParcelServiceService $service
    ) {
        $this->requestParser    = $requestParser;
        $this->requestValidator = $requestValidator;
        $this->service          = $service;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $referencedErrors = $this->requestValidator->validatePostRequestBody($request->getParsedBody());
        if (count($referencedErrors) > 0) {
            return $response->withStatus(400)->withJson(['errors' => $referencedErrors]);
        }
        
        $parcelServices = [];
        foreach ($request->getParsedBody() as $reference => $details) {
            try {
                $parcelServices[$reference] = ParcelService::createFromArray($details);
            } catch (InvalidArgumentException $exception) {
                $referencedErrors[$reference][] = $exception->getMessage();
            }
        }
        if (count($referencedErrors) > 0) {
            return $response->withStatus(422)->withJson(['errors' => $referencedErrors]);
        }
        
        $ids   = $this->service->storeParcelServices(ParcelServices::createWithReferences($parcelServices));
        $links = [];
        /** @var ParcelServiceId $id */
        foreach ($ids as $reference => $id) {
            $links[(string)$reference] = rtrim($this->requestParser->getResourceUrlFromRequest($request), '/')
                                         . '/parcel-services/' . $id->value();
        }
        
        $metaData = $this->createApiMetaData($links);
        
        return $response->withJson([
                                       'data'  => $ids,
                                       '_meta' => $metaData,
                                   ]);
    }
}